﻿using Google.Apis.Auth.OAuth2;
using Google.Apis.Util.Store;
using MailKit.Net.Smtp;
using MailKit.Security;
using MimeKit;
using Nova.Models.Correo;
using System;
using System.IO;
using System.Net;
using System.Security.Authentication;
using System.Threading;
using System.Threading.Tasks;
using System.Web.Hosting;

public class CorreoService
{
    private const string SMTP_HOST = "smtp.gmail.com";
    private const int SMTP_PORT = 587;
    private const string SMTP_USERNAME = "sistemas@cecyteo.edu.mx";

    // Rutas para OAuth
    private static readonly string[] Scopes = { "https://mail.google.com/" };
    private static readonly string ApplicationName = "Sistema MVC Correo";
    private static readonly string CredentialsPath = HostingEnvironment.MapPath("~/App_Data/credentials.json");
    private static readonly string TokenPath = HostingEnvironment.MapPath("~/App_Data/token/");

    public static void EnviarCorreoPersonalizado(CorreoPersonalizado correoPersonalizado, string rutaHtml)
    {
        try
        {
            // Llamar a la versión asíncrona y esperar su resultado
            EnviarCorreoPersonalizadoAsync(correoPersonalizado, rutaHtml).GetAwaiter().GetResult();
        }
        catch (Exception ex)
        {
            throw new Exception("Error al enviar correo: " + ex.Message, ex);
        }
    }

    public static async Task EnviarCorreoPersonalizadoAsync(CorreoPersonalizado correoPersonalizado, string rutaHtml)
    {
        try
        {
            // Preparar el mensaje con MimeKit
            var message = new MimeMessage();
            message.From.Add(new MailboxAddress("Sistema", SMTP_USERNAME));
            message.To.Add(new MailboxAddress(correoPersonalizado.Nombre, correoPersonalizado.DestinatarioEmail));
            message.Subject = correoPersonalizado.TituloCorreo;

            // Obtener y establecer el cuerpo HTML
            string body = ObtenerPlantillaHTML(correoPersonalizado, rutaHtml);
            var bodyBuilder = new BodyBuilder
            {
                HtmlBody = body
            };
            message.Body = bodyBuilder.ToMessageBody();

            // Obtener credenciales OAuth
            var credential = await ObtenerCredencialOAuthAsync();

            using (var client = new SmtpClient())
            {
                try
                {
                    client.ServerCertificateValidationCallback = (sender, certificate, chain, errors) => true;
                    client.SslProtocols = System.Security.Authentication.SslProtocols.Tls12;
                    client.Connect(SMTP_HOST, SMTP_PORT, SecureSocketOptions.StartTls);
                    var oauth2 = new SaslMechanismOAuth2(SMTP_USERNAME, credential.Token.AccessToken);
                    client.Authenticate(oauth2);
                    client.Send(message);
                    client.Disconnect(true);
                }
                catch (Exception ex)
                {
                    throw new Exception("Error de conexión: " + ex.Message +
                        (ex.InnerException != null ? " - Detalle: " + ex.InnerException.Message : ""));
                }

            }

        }
        catch (Exception ex)
        {
            var msg = "Error al enviar correo: " + ex.Message;
            if (ex.InnerException != null)
            {
                msg += " Detalle: " + ex.InnerException.Message;
            }
            throw new Exception(msg);
        }
    }


    private static async Task<UserCredential> ObtenerCredencialOAuthAsync()
    {
        try
        {
            // 1. Validar que credentials.json existe
            if (!File.Exists(CredentialsPath))
                throw new FileNotFoundException("No se encontró el archivo de credenciales.", CredentialsPath);

            // 2. Validar que la carpeta token/ existe
            if (!Directory.Exists(TokenPath))
            {
                Directory.CreateDirectory(TokenPath);
            }

            // 3. Validar que exista el archivo real del token
            var tokenFilePath = Path.Combine(TokenPath, "Google.Apis.Auth.OAuth2.Responses.TokenResponse-sistemas@cecyteo.edu.mx");
            if (!File.Exists(tokenFilePath))
            {
                if (File.Exists("token.json"))
                {
                    // Copiar manualmente si existe token.json
                    File.Copy("token.json", tokenFilePath);
                }
                else
                {
                    throw new FileNotFoundException("No se encontró el archivo de token necesario para autenticación OAuth.", tokenFilePath);
                }
            }

            // 4. Continuar con carga de credenciales
            UserCredential credential;
            using (var stream = new FileStream(CredentialsPath, FileMode.Open, FileAccess.Read))
            {
                var clientSecrets = GoogleClientSecrets.FromStream(stream).Secrets;
                var dataStore = new FileDataStore(TokenPath, true);

                credential = await GoogleWebAuthorizationBroker.AuthorizeAsync(
                    clientSecrets,
                    Scopes,
                    SMTP_USERNAME, // Aquí mejor usar directamente SMTP_USERNAME
                    CancellationToken.None,
                    dataStore);
            }

            // 5. Verificar si el token expiró
            var vencimiento = credential.Token.IssuedUtc + TimeSpan.FromSeconds(credential.Token.ExpiresInSeconds ?? 0);
            if (DateTime.UtcNow >= vencimiento)
            {
                if (!await credential.RefreshTokenAsync(CancellationToken.None))
                {
                    // Si no se pudo refrescar, borrar y reautorizar
                    if (File.Exists(tokenFilePath))
                        File.Delete(tokenFilePath);

                    using (var stream = new FileStream(CredentialsPath, FileMode.Open, FileAccess.Read))
                    {
                        var refreshedClientSecrets = GoogleClientSecrets.FromStream(stream).Secrets;
                        var dataStore = new FileDataStore(TokenPath, true);

                        credential = await GoogleWebAuthorizationBroker.AuthorizeAsync(
                            refreshedClientSecrets,
                            Scopes,
                            SMTP_USERNAME,
                            CancellationToken.None,
                            dataStore);
                    }
                }
            }

            return credential;
        }
        catch (Exception ex)
        {
            throw new Exception("Error al obtener credenciales OAuth: " + ex.Message, ex);
        }
    }

    private static string ObtenerPlantillaHTML(CorreoPersonalizado correoPersonalizado, string rutaPlantilla)
    {
        // Obtén la ruta física del archivo a partir de la ruta relativa
        string rutaFisicaPlantilla = HostingEnvironment.MapPath(rutaPlantilla);

        // Lee el contenido del archivo HTML
        string plantillaHTML = File.ReadAllText(rutaFisicaPlantilla);

        // Reemplaza las variables con los datos personalizados
        plantillaHTML = plantillaHTML.Replace("{{Nombre}}", correoPersonalizado.Nombre)
            .Replace("{{LugarTrabajo}}", correoPersonalizado.LugarTrabajo)
            .Replace("{{Firma}}", correoPersonalizado.Firma)
            .Replace("{{DirigidoA}}", correoPersonalizado.DirigidoA)
            .Replace("{{DireccionFirma}}", correoPersonalizado.DireccionFirma)
            .Replace("{{Fecha}}", correoPersonalizado.Fecha)
            .Replace("{{Hora}}", correoPersonalizado.Hora)
            .Replace("{{LinkUrl}}", correoPersonalizado.LinkUrl);

        return plantillaHTML;
    }

    public static async void EnviarCorreoPersonalizadoConCuerpoDirecto(CorreoPersonalizado correoPersonalizado, string cuerpoHtml)
    {
        try
        {
            // Preparar mensaje
            var message = new MimeMessage();
            message.From.Add(new MailboxAddress("Sistema", SMTP_USERNAME));
            message.To.Add(new MailboxAddress(correoPersonalizado.Nombre, correoPersonalizado.DestinatarioEmail));
            message.Subject = correoPersonalizado.TituloCorreo;

            var bodyBuilder = new BodyBuilder
            {
                HtmlBody = cuerpoHtml
            };
            message.Body = bodyBuilder.ToMessageBody();

            // Obtener credenciales OAuth
            // Obtener credenciales OAuth
            var credential = await ObtenerCredencialOAuthAsync();
            //var credential = ObtenerCredencialOAuthAsync().GetAwaiter().GetResult();

            using (var client = new SmtpClient())
            {
                client.ServerCertificateValidationCallback = (sender, certificate, chain, errors) => true;
                client.SslProtocols = System.Security.Authentication.SslProtocols.Tls12;
                client.Connect(SMTP_HOST, SMTP_PORT, SecureSocketOptions.StartTls);

                var oauth2 = new SaslMechanismOAuth2(SMTP_USERNAME, credential.Token.AccessToken);
                client.Authenticate(oauth2);

                client.Send(message);
                client.Disconnect(true);
            }
        }
        catch (Exception ex)
        {
            throw new Exception("Error al enviar correo: " + ex.Message, ex);
        }
    }


}