﻿using Dapper;
using Nova.Libraries;
using Nova.Models.Designpattern;
using System;
using System.Collections.Generic;
using System.Data.Entity;
using System.Linq;
using System.Threading.Tasks;
using System.Web;

namespace Nova.Models.Nomina
{
    // Modelo de error para capturar los detalles de las validaciones
    public class ErrorNomina
    {
        public int IdTrabajador { get; set; }
        public string TrabajadorNombre { get; set; }
        public string Campo { get; set; }
        public string Nomina { get; set; }
        public string RelacionLaboral { get; set; }
        public string DetalleError { get; set; }
        public decimal ValorActual { get; set; }
        public decimal ValorEsperado { get; set; }
    }

    public class NominaValidator
    {
        private const decimal MaxPercentageDeviation = 0.10m;  // 40% de desviación permitida
        private const int MaxHorasSemanaMes = 40; // Límite de horas por semana

        // Método para validar percepciones de sueldo bruto y devolver errores en una lista
        public List<ErrorNomina> ValidateNomina(List<Nomina_stNominaTrabajador> nomina)
        {
            NominaHelper nominaHelper = new NominaHelper();
            List<ErrorNomina> errores = new List<ErrorNomina>();

            if (nomina.Count() > 0)
            {
                var registro = nomina.FirstOrDefault();
                var resultado1 = nominaHelper.ObtenerQuincenaAnterior(
                    Nova.Libraries.ConvertToInt.ConvertToInt16(registro.Nomina_stNomina.Catalogos_ctQuincenaMes.QuincenaMes),
                    registro.Nomina_stNomina.Año
                );

                using (var bd = new bdNovaEntities())
                {
                    string qna = resultado1.quincenaAnterior.ToString("d2");
                    var trabajadoresAnteriores = bd.Nomina_stNominaTrabajador.AsNoTracking()
                        .Where(z => z.Nomina_stNomina.Año == resultado1.anioAnterior &&
                                    z.Nomina_stNomina.Catalogos_ctQuincenaMes.QuincenaMes == qna &&
                                    z.Nomina_stNomina.IdPlantillaNomina == registro.Nomina_stNomina.IdPlantillaNomina)
                        .Include(z => z.Convenios_stConvenioLaboralTrabajadorCategoria)
                        .Include(z => z.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador)
                        .Include(z => z.Nomina_stNomina.Nomina_ctPlantillaNomina.Convenios_stRelacionConvenioLaboral)
                        .ToList(); // Usamos ToList() en lugar de ToListAsync()

                    // Crear una lista de tareas para validar cada trabajador de manera paralela
                    var tareas = nomina.Select(trabajador => Task.Run(() =>
                    {
                        var anterior = trabajadoresAnteriores
                            .FirstOrDefault(a => a.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.IdTrabajador
                                                 == trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.IdTrabajador);

                        if (anterior != null)
                        {
                            if (trabajador.SueldoBruto > 0)
                            {
                                // Validar sueldo bruto
                                decimal desviacionSueldo = Math.Abs(trabajador.SueldoBruto - anterior.SueldoBruto) / trabajador.SueldoBruto;
                                if (desviacionSueldo > MaxPercentageDeviation)
                                {
                                    lock (errores)
                                    {
                                        errores.Add(new ErrorNomina
                                        {
                                            TrabajadorNombre = $"{trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.Nombre} {trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.ApellidoPaterno} {trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.ApellidoMaterno}",
                                            IdTrabajador = trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.IdTrabajador,
                                            Nomina = trabajador.Nomina_stNomina.Nomina_ctPlantillaNomina.Plantilla,
                                            RelacionLaboral = trabajador.Nomina_stNomina.Nomina_ctPlantillaNomina.Convenios_stRelacionConvenioLaboral.RelacionConvenioLaboral,
                                            Campo = "Sueldo Brruto",
                                            DetalleError = $"Desviación mayor al {(MaxPercentageDeviation * 100)}% comparado con la quincena anterior",
                                            ValorActual = trabajador.SueldoBruto,
                                            ValorEsperado = anterior.SueldoBruto
                                        });
                                    }
                                }
                            }
                            else {

                                lock (errores)
                                {
                                    errores.Add(new ErrorNomina
                                    {
                                        TrabajadorNombre = $"{trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.Nombre} {trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.ApellidoPaterno} {trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.ApellidoMaterno}",
                                        IdTrabajador = trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.IdTrabajador,
                                        Nomina = trabajador.Nomina_stNomina.Nomina_ctPlantillaNomina.Plantilla,
                                        RelacionLaboral = trabajador.Nomina_stNomina.Nomina_ctPlantillaNomina.Convenios_stRelacionConvenioLaboral.RelacionConvenioLaboral,
                                        Campo = "Sueldo Neto",
                                        DetalleError = $"SUELDO EN 0",
                                        ValorActual = trabajador.SueldoBruto,
                                        ValorEsperado = anterior.SueldoBruto
                                    });
                                }
                            }
                            // Validar horas trabajadas
                            if (trabajador.HorasSemanaMes > 0 && anterior.HorasSemanaMes != trabajador.HorasSemanaMes)
                            {
                                decimal desviacionHoras = Math.Abs(trabajador.HorasSemanaMes - anterior.HorasSemanaMes) / trabajador.HorasSemanaMes;
                                if (desviacionHoras > MaxPercentageDeviation)
                                {
                                    lock (errores)
                                    {
                                        errores.Add(new ErrorNomina
                                        {
                                            TrabajadorNombre = $"{trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.Nombre} {trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.ApellidoPaterno} {trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.Kardex_ctTrabajador.Kardex_ctDatosPersonales.ApellidoMaterno}",
                                            IdTrabajador = trabajador.Convenios_stConvenioLaboralTrabajadorCategoria.Convenios_stConvenioLaboralTrabajador.IdTrabajador,
                                            Nomina = trabajador.Nomina_stNomina.Nomina_ctPlantillaNomina.Plantilla,
                                            RelacionLaboral = trabajador.Nomina_stNomina.Nomina_ctPlantillaNomina.Convenios_stRelacionConvenioLaboral.RelacionConvenioLaboral,
                                            Campo = "Horas Trabajadas",
                                            DetalleError = $"Desviación mayor al {(MaxPercentageDeviation * 100)}% comparado con la quincena anterior",
                                            ValorActual = trabajador.HorasSemanaMes,
                                            ValorEsperado = anterior.HorasSemanaMes
                                        });
                                    }
                                }
                            }
                        }
                    })).ToList();

                    // Esperar a que todas las tareas terminen
                    Task.WaitAll(tareas.ToArray());

                   
                }
            }
            return errores;
        }
        }
    }


